<?php

namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Order;
use App\Services\Payment\RazorpayService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class OrderController extends Controller
{
    protected $razorpay;

    public function __construct(RazorpayService $razorpay)
    {
        $this->razorpay = $razorpay;
    }

    public function checkout(Request $request)
    {
        $request->validate(['product_id' => 'required|exists:products,id']);

        $product = Product::findOrFail($request->product_id);

        // internal: Create pending order
        $order = Order::create([
            'client_id' => Auth::id(),
            'amount' => $product->monthly_price,
            'status' => 'pending',
            'payment_method' => 'razorpay'
        ]);

        try {
            // razorpay: Create order on gateway
            $razorpayOrder = $this->razorpay->createOrder($order->amount, $order->id);

            // Update local order with Razorpay ID
            $order->update(['transaction_id' => $razorpayOrder['id']]);

            return view('client.checkout', compact('order', 'product', 'razorpayOrder'));
        } catch (\Exception $e) {
            Log::error("Checkout Failed: " . $e->getMessage());
            return back()->with('error', 'Unable to initiate payment. Please try again.');
        }
    }

    public function verifyPayment(Request $request)
    {
        $success = $this->razorpay->verifyPaymentSignature($request->all());

        if ($success) {
            $order = Order::where('transaction_id', $request->razorpay_order_id)->first();

            if ($order) {
                $order->update(['status' => 'active']);

                // Dispatch job to create hosting account
                \App\Jobs\CreateHostingAccount::dispatch($order);

                return redirect()->route('client.dashboard')->with('success', 'Payment Successful! Service is provisioning.');
            }
        }

        return back()->with('error', 'Payment Verification Failed');
    }
}
