<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Services\Installer\InstallerService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Exception;

class InstallerController extends Controller
{
    protected $installer;

    public function __construct(InstallerService $installer)
    {
        $this->installer = $installer;
    }

    public function welcome()
    {
        return view('installer.welcome');
    }

    public function requirements()
    {
        $requirements = $this->installer->checkRequirements();
        $allMet = true;

        foreach ($requirements['php'] as $key => $val) {
            if ($key === 'status' && !$val)
                $allMet = false;
        }
        foreach ($requirements['extensions'] as $ext => $enabled) {
            if (!$enabled)
                $allMet = false;
        }
        foreach ($requirements['permissions'] as $path => $writable) {
            if (!$writable)
                $allMet = false;
        }

        return view('installer.requirements', compact('requirements', 'allMet'));
    }

    public function license()
    {
        return view('installer.license');
    }

    public function verifyLicense(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'purchase_code' => 'required|string',
            'username' => 'required|string',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Mock License Verification
        // In real world, call API
        $valid = true;

        if ($valid) {
            $request->session()->put('installer.license', $request->only('purchase_code', 'username'));
            Log::info('Installer: License verified for user ' . $request->username);
            return redirect()->route('installer.database');
        }

        return back()->withErrors(['purchase_code' => 'Invalid license key.']);
    }

    public function database()
    {
        return view('installer.database');
    }

    public function saveDatabase(Request $request)
    {
        $credentials = $request->validate([
            'host' => 'required|string',
            'port' => 'required|numeric',
            'database' => 'required|string',
            'username' => 'required|string',
            'password' => 'nullable|string',
        ]);

        if (!$this->installer->checkDatabaseConnection($credentials)) {
            return back()->withErrors(['connection' => 'Could not connect to the database. Check credentials.'])->withInput();
        }

        $this->installer->updateEnv([
            'DB_HOST' => $credentials['host'],
            'DB_PORT' => $credentials['port'],
            'DB_DATABASE' => $credentials['database'],
            'DB_USERNAME' => $credentials['username'],
            'DB_PASSWORD' => $credentials['password'] ?? '',
        ]);

        Log::info('Installer: Database configuration saved.');

        return redirect()->route('installer.app');
    }

    public function appSetup()
    {
        return view('installer.app_setup');
    }

    public function saveApp(Request $request)
    {
        $data = $request->validate([
            'app_name' => 'required|string',
            'app_url' => 'required|url',
        ]);

        $this->installer->updateEnv([
            'APP_NAME' => $data['app_name'],
            'APP_URL' => $data['app_url'],
        ]);

        Log::info('Installer: App configuration saved.');

        return redirect()->route('installer.admin');
    }

    public function adminSetup()
    {
        return view('installer.admin_setup');
    }

    public function saveAdmin(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string',
            'email' => 'required|email',
            'password' => 'required|confirmed|min:8',
        ]);

        $request->session()->put('installer.admin', $data);

        Log::info('Installer: Admin details collected.');

        return view('installer.process');
    }

    public function install(Request $request)
    {
        try {
            Log::info('Installer: Starting installation process...');

            // 0. Ensure Config is fresh (Check connection)
            // We assume .env was updated and picked up.
            // If connection fails here, it throws exception which is caught.

            // 1. Migrate
            Log::info('Installer: Running migrations...');
            Artisan::call('migrate:fresh', ['--force' => true]);

            // 2. Custom Seeders (Roles, etc) if any
            // Artisan::call('db:seed', ['--force' => true]);

            // 3. Create Admin
            Log::info('Installer: Creating admin user...');
            $adminData = $request->session()->get('installer.admin');
            if (!$adminData) {
                throw new Exception("Session expired. Please start over.");
            }

            $user = User::create([
                'name' => $adminData['name'],
                'email' => $adminData['email'],
                'password' => Hash::make($adminData['password']),
                'email_verified_at' => now(),
            ]);

            // 4. Save License (Mock)
            Log::info('Installer: Saving license...');
            $licenseData = $request->session()->get('installer.license');
            if ($licenseData) {
                DB::table('licenses')->insert([
                    'key' => encrypt($licenseData['purchase_code']),
                    'domain' => request()->getHost(),
                    'status' => 'active',
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            // 5. Create Default Tenant (SaaS Prep)
            Log::info('Installer: Creating default tenant...');
            DB::table('tenants')->insert([
                'name' => 'Default Tenant',
                'domain' => request()->getHost(),
                'database_name' => DB::connection()->getDatabaseName(),
                'status' => 'active',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // 6. Security & Cleanup
            Log::info('Installer: Locking installation...');
            file_put_contents(storage_path('installed.lock'), 'INSTALLED on ' . now());

            Artisan::call('config:clear');
            Artisan::call('cache:clear');
            Artisan::call('route:clear'); // Clear route cache too

            Log::info('Installer: Installation completed successfully.');

            return response()->json(['status' => 'success', 'redirect' => route('installer.finish')]);

        } catch (Exception $e) {
            Log::error('Installer Error: ' . $e->getMessage());
            return response()->json(['status' => 'error', 'message' => $e->getMessage()], 500);
        }
    }

    public function finish()
    {
        return view('installer.finish');
    }
}
