<?php

namespace App\Services\Installer;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

class InstallerService
{
    /**
     * Update the .env file with given key-value pairs.
     *
     * @param array $data
     * @return bool
     */
    public function updateEnv(array $data)
    {
        $envPath = base_path('.env');

        if (!File::exists($envPath)) {
            return false;
        }

        $envContent = File::get($envPath);

        foreach ($data as $key => $value) {
            // Wrap string with quotes if it contains spaces
            if (str_contains($value, ' ')) {
                $value = '"' . $value . '"';
            }

            // Check if key exists
            if (preg_match("/^{$key}=/m", $envContent)) {
                // Update existing key
                $envContent = preg_replace("/^{$key}=.*/m", "{$key}={$value}", $envContent);
            } else {
                // Append new key
                $envContent .= "\n{$key}={$value}";
            }
        }

        return File::put($envPath, $envContent);
    }

    /**
     * Check database connection with provided credentials.
     *
     * @param array $credentials
     * @return bool
     */
    public function checkDatabaseConnection(array $credentials)
    {
        try {
            // Create a temporary connection configuration
            config([
                'database.connections.installer_temp' => [
                    'driver' => 'mysql',
                    'host' => $credentials['host'],
                    'port' => $credentials['port'],
                    'database' => $credentials['database'],
                    'username' => $credentials['username'],
                    'password' => $credentials['password'],
                    'charset' => 'utf8mb4',
                    'collation' => 'utf8mb4_unicode_ci',
                    'prefix' => '',
                    'strict' => true,
                    'engine' => null,
                ]
            ]);

            DB::connection('installer_temp')->getPdo();

            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Check server requirements.
     *
     * @return array
     */
    public function checkRequirements()
    {
        $requirements = [
            'php' => [
                'version' => '8.2.0',
                'current' => phpversion(),
                'status' => version_compare(phpversion(), '8.2.0', '>='),
            ],
            'extensions' => [
                'pdo' => extension_loaded('pdo'),
                'pdo_mysql' => extension_loaded('pdo_mysql'),
                'curl' => extension_loaded('curl'),
                'openssl' => extension_loaded('openssl'),
                'mbstring' => extension_loaded('mbstring'),
                'tokenizer' => extension_loaded('tokenizer'),
                'xml' => extension_loaded('xml'),
                'fileinfo' => extension_loaded('fileinfo'),
                'json' => extension_loaded('json'),
            ],
            'permissions' => [
                'storage' => is_writable(storage_path()),
                'bootstrap/cache' => is_writable(base_path('bootstrap/cache')),
                '.env' => is_writable(base_path('.env')),
            ]
        ];

        return $requirements;
    }
}
