<?php

namespace App\Services\Provisioning;

use Illuminate\Support\Facades\Http;
use Exception;
use Illuminate\Support\Facades\Log;

class WhmService
{
    protected string $whmUrl;
    protected string $whmUser;
    protected string $whmToken;

    public function __construct()
    {
        $this->whmUrl = config('services.whm.url');
        $this->whmUser = config('services.whm.username');
        $this->whmToken = config('services.whm.token');
    }

    /**
     * Create a new cPanel account
     */
    public function createAccount(array $data)
    {
        // WHM API: createacct
        $response = Http::withHeaders([
            'Authorization' => 'whm ' . $this->whmUser . ':' . $this->whmToken
        ])->withoutVerifying() // Optional: remove if SSL is valid on WHM server
            ->get($this->whmUrl . '/json-api/createacct', [
                'api.version' => 1,
                'username' => $data['username'],
                'domain' => $data['domain'],
                'password' => $data['password'],
                'plan' => $data['package_name'],
                'contactemail' => $data['email'],
            ]);

        $this->logResponse('createacct', $response);

        if ($response->json('metadata.result') === 0) {
            throw new Exception("WHM Error: " . $response->json('metadata.reason'));
        }

        return $response->json();
    }

    public function suspendAccount(string $username, string $reason = 'Overdue Payment')
    {
        $response = $this->callWhm('suspendacct', [
            'user' => $username,
            'reason' => $reason
        ]);

        if ($response['metadata']['result'] === 0) {
            Log::error("WHM Suspend Failed: " . $response['metadata']['reason']);
        }

        return $response;
    }

    public function unsuspendAccount(string $username)
    {
        return $this->callWhm('unsuspendacct', ['user' => $username]);
    }

    public function terminateAccount(string $username)
    {
        return $this->callWhm('removeacct', ['user' => $username]);
    }

    protected function callWhm($function, $params = [])
    {
        $params['api.version'] = 1;

        $response = Http::withHeaders([
            'Authorization' => 'whm ' . $this->whmUser . ':' . $this->whmToken
        ])->withoutVerifying()
            ->get($this->whmUrl . '/json-api/' . $function, $params);

        $this->logResponse($function, $response);

        return $response->json();
    }

    protected function logResponse($function, $response)
    {
        if ($response->failed() || $response->json('metadata.result') === 0) {
            Log::error("WHM API [$function] Failed: " . $response->body());
        } else {
            Log::info("WHM API [$function] Success");
        }
    }
}
