<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Clients Table
        Schema::create('clients', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('email')->unique();
            $table->string('password');
            $table->string('phone')->nullable();
            $table->string('company')->nullable();
            $table->text('address')->nullable();
            $table->string('city')->nullable();
            $table->string('state')->nullable();
            $table->string('country')->nullable();
            $table->string('postcode')->nullable();
            $table->string('currency')->default('USD');
            $table->enum('status', ['active', 'inactive', 'closed'])->default('active');
            $table->decimal('credit', 10, 2)->default(0.00);
            $table->timestamp('email_verified_at')->nullable();
            $table->rememberToken();
            $table->timestamps();
        });

        // Products (Hosting Plans)
        Schema::create('products', function (Blueprint $table) {
            $table->id();
            $table->string('name'); // e.g., "Basic Plan"
            $table->text('description')->nullable();
            $table->decimal('monthly_price', 10, 2)->nullable();
            $table->decimal('annual_price', 10, 2)->nullable();
            $table->string('whm_package_name')->nullable(); // Matches WHM Package Name
            $table->unsignedBigInteger('server_group_id')->nullable();
            $table->boolean('is_hidden')->default(false);
            $table->integer('sort_order')->default(0);
            $table->timestamps();
        });

        // Servers (cPanel/WHM)
        Schema::create('servers', function (Blueprint $table) {
            $table->id();
            $table->string('name'); // e.g., "Server 1"
            $table->string('hostname'); // e.g., "server1.luckyscloud.com"
            $table->string('ip_address');
            $table->string('whm_user'); // "root" or reseller user
            $table->text('whm_token'); // Encrypted API Token
            $table->integer('max_accounts')->default(100);
            $table->boolean('active')->default(true);
            $table->timestamps();
        });

        // Orders
        Schema::create('orders', function (Blueprint $table) {
            $table->id();
            $table->foreignId('client_id')->constrained()->onDelete('cascade');
            $table->decimal('amount', 10, 2);
            $table->string('payment_method')->default('razorpay');
            $table->enum('status', ['pending', 'active', 'fraud', 'cancelled'])->default('pending');
            $table->string('transaction_id')->nullable(); // Gateway Trans ID
            $table->ipAddress('ip_address')->nullable();
            $table->timestamps();
        });

        // Invoices
        Schema::create('invoices', function (Blueprint $table) {
            $table->id();
            $table->foreignId('client_id')->constrained()->onDelete('cascade');
            $table->string('invoice_number')->unique(); // e.g., INV-2023-0001
            $table->date('date');
            $table->date('due_date');
            $table->timestamp('paid_at')->nullable();
            $table->enum('status', ['unpaid', 'paid', 'cancelled', 'refunded'])->default('unpaid');
            $table->decimal('total', 10, 2);
            $table->decimal('subtotal', 10, 2);
            $table->decimal('tax', 10, 2)->default(0);
            $table->string('payment_method')->default('razorpay');
            $table->timestamps();
        });

        // Invoice Items
        Schema::create('invoice_items', function (Blueprint $table) {
            $table->id();
            $table->foreignId('invoice_id')->constrained()->onDelete('cascade');
            $table->foreignId('service_id')->nullable(); // Link to service if renewal
            $table->text('description');
            $table->decimal('amount', 10, 2);
            $table->boolean('taxed')->default(false);
            $table->timestamps();
        });

        // Services (Hosting Accounts)
        Schema::create('services', function (Blueprint $table) {
            $table->id();
            $table->foreignId('client_id')->constrained()->onDelete('cascade');
            $table->foreignId('order_id')->nullable();
            $table->foreignId('product_id')->constrained();
            $table->foreignId('server_id')->nullable()->constrained();

            $table->string('domain')->nullable();
            $table->string('username')->nullable(); // cPanel username
            $table->text('password')->nullable(); // Encrypted cPanel password

            $table->enum('billing_cycle', ['monthly', 'quarterly', 'semi_annually', 'annually', 'biennially', 'triennially']);
            $table->decimal('recurring_amount', 10, 2);

            $table->date('next_due_date')->nullable();
            $table->enum('status', ['pending', 'active', 'suspended', 'terminated', 'cancelled', 'fraud'])->default('pending');
            $table->text('suspend_reason')->nullable();

            $table->timestamps();
        });

        // Transactions
        Schema::create('transactions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('client_id')->constrained();
            $table->foreignId('invoice_id')->constrained();
            $table->string('gateway'); // razorpay
            $table->string('transaction_id'); // Gateway ID
            $table->decimal('amount_in', 10, 2)->default(0);
            $table->decimal('amount_out', 10, 2)->default(0);
            $table->text('description')->nullable();
            $table->timestamps();
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('transactions');
        Schema::dropIfExists('services');
        Schema::dropIfExists('invoice_items');
        Schema::dropIfExists('invoices');
        Schema::dropIfExists('orders');
        Schema::dropIfExists('servers');
        Schema::dropIfExists('products');
        Schema::dropIfExists('clients');
    }
};
