<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('services', function (Blueprint $table) {
            // SaaS & User refactor
            if (!Schema::hasColumn('services', 'tenant_id')) {
                $table->unsignedBigInteger('tenant_id')->nullable()->after('id')->index();
                // If tenants table exists, add constraint. 
                // We use check to avoid error if tenants migration ran after (though timestamp suggests before).
                if (Schema::hasTable('tenants')) {
                    $table->foreign('tenant_id')->references('id')->on('tenants')->nullOnDelete();
                }
            }

            if (!Schema::hasColumn('services', 'user_id')) {
                $table->foreignId('user_id')->nullable()->after('tenant_id')->constrained('users')->cascadeOnDelete();
            }

            // Make product_id nullable if requested or leave as is. 
            // User requested: "product_id (nullable for now)"
            // It is currently constrained. To make nullable, we need to modify it.
            $table->foreignId('product_id')->nullable()->change();

            // Add suspended_at, terminated_at if missing
            if (!Schema::hasColumn('services', 'suspended_at')) {
                $table->timestamp('suspended_at')->nullable();
            }
            if (!Schema::hasColumn('services', 'terminated_at')) {
                $table->timestamp('terminated_at')->nullable();
            }

            // Drop client_id if exists
            if (Schema::hasColumn('services', 'client_id')) {
                // Drop foreign key first if it exists. 
                // Naming convention: services_client_id_foreign
                $table->dropForeign(['client_id']);
                $table->dropColumn('client_id');
            }
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('services', function (Blueprint $table) {
            $table->unsignedBigInteger('client_id')->nullable();
            // Re-add foreign key if needed, or just leave as nullable integer for rollback safety

            $table->dropForeign(['user_id']);
            $table->dropColumn('user_id');

            $table->dropForeign(['tenant_id']);
            $table->dropColumn('tenant_id');

            $table->dropColumn(['suspended_at', 'terminated_at']);
        });
    }
};
